class Card {
    constructor(rank, suit) {
        this.rank = rank;
        this.suit = suit;
    }

    getValue() {
        if (['J', 'Q', 'K'].includes(this.rank)) return 10;
        if (this.rank === 'A') return 11; 
        return parseInt(this.rank);
    }
}

class Deck {
    constructor() {
        this.cards = [];
        const suits = ['Hearts', 'Diamonds', 'Clubs', 'Spades'];
        const ranks = ['2', '3', '4', '5', '6', '7', '8', '9', '10', 'J', 'Q', 'K', 'A'];

        for (const suit of suits) {
            for (const rank of ranks) {
                this.cards.push(new Card(rank, suit));
            }
        }
        this.shuffle();
    }

    shuffle() {
        for (let i = this.cards.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [this.cards[i], this.cards[j]] = [this.cards[j], this.cards[i]];
        }
    }

    drawCard() {
        return this.cards.pop();
    }
}

class Hand {
    constructor() {
        this.cards = [];
    }

    addCard(card) {
        this.cards.push(card);
    }

    getValue() {
        let value = this.cards.reduce((acc, card) => acc + card.getValue(), 0);
        let aceCount = this.cards.filter(card => card.rank === 'A').length;

        while (value > 21 && aceCount > 0) {
            value -= 10; 
            aceCount--;
        }
        return value;
    }

    showHand() {
        return this.cards.map(card => `${card.rank} of ${card.suit}`).join(', ');
    }
}

let deck, playerHand, dealerHand;
let playerMoney = 100; // Starting money

const dealerSection = document.getElementById('dealer-hand');
const playerSection = document.getElementById('player-hand');
const dealerScore = document.getElementById('dealer-score');
const playerScore = document.getElementById('player-score');
const message = document.getElementById('message');

document.getElementById('start-button').addEventListener('click', startGame);
document.getElementById('hit-button').addEventListener('click', playerHit);
document.getElementById('stand-button').addEventListener('click', dealerTurn);

function startGame() {
    const betAmount = parseInt(document.getElementById('bet-amount').value);
    if (betAmount > playerMoney || betAmount <= 0) {
        alert("Invalid bet amount!");
        return;
    }
    playerMoney -= betAmount;
    document.getElementById('bet-amount').value = '';

    deck = new Deck();
    playerHand = new Hand();
    dealerHand = new Hand();

    playerHand.addCard(deck.drawCard());
    playerHand.addCard(deck.drawCard());
    dealerHand.addCard(deck.drawCard());
    dealerHand.addCard(deck.drawCard());

    updateDisplay();
    document.getElementById('hit-button').disabled = false;
    document.getElementById('stand-button').disabled = false;
    document.getElementById('start-button').disabled = true;
}

function playerHit() {
    playerHand.addCard(deck.drawCard());
    if (playerHand.getValue() > 21) {
        endGame("You bust! Dealer wins.");
    }
    updateDisplay();
}

function dealerTurn() {
    while (dealerHand.getValue() < 17) {
        dealerHand.addCard(deck.drawCard());
    }
    determineWinner();
}

function updateDisplay() {
    playerSection.innerHTML = playerHand.showHand();
    playerScore.innerText = `Value: ${playerHand.getValue()}`;
    dealerSection.innerHTML = dealerHand.showHand().replace(/^[^,]*/, 'Hidden Card,');
    dealerScore.innerText = `Value: ?`;
}

function determineWinner() {
    dealerSection.innerHTML = dealerHand.showHand();
    dealerScore.innerText = `Value: ${dealerHand.getValue()}`;

    const playerValue = playerHand.getValue();
    const dealerValue = dealerHand.getValue();

    if (playerValue > 21) {
        endGame("You bust! Dealer wins.");
    } else if (dealerValue > 21 || playerValue > dealerValue) {
        endGame("You win!");
        playerMoney += parseInt(document.getElementById('bet-amount').value) * 2; // Win bet
    } else if (playerValue < dealerValue) {
        endGame("Dealer wins!");
    } else {
        endGame("It's a tie!");
        playerMoney += parseInt(document.getElementById('bet-amount').value); // Return bet
    }
}

function endGame(result) {
    message.innerText = result;
    document.getElementById('hit-button').disabled = true;
    document.getElementById('stand-button').disabled = true;
    document.getElementById('start-button').disabled = false;
    document.getElementById('bet-amount').disabled = false;

    // Show remaining money
    message.innerText += ` Your remaining money: $${playerMoney}`;
}
